<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\UserLedger;
use App\Models\Deposit;
use Carbon\Carbon;

class WeeklySalaryCommand extends Command
{
    protected $signature = 'salary:weekly';
    protected $description = 'Distribute weekly salaries to qualified users';

    public function handle()
    {
        $users = User::all();
        $today = Carbon::now()->startOfDay();

        $this->info("Weekly salary process started on {$today->toDateString()}");

        foreach ($users as $user) {
            // Count valid referred users with approved deposits
            $validInvites = User::where('ref_by', $user->ref_id)
                ->whereHas('deposits', function ($query) {
                    $query->where('status', 'approved');
                })
                ->count();

            $salary = 0;
            if ($validInvites >= 19) {
                $salary = 500;
            } elseif ($validInvites >= 6) {
                $salary = 250;
            } elseif ($validInvites >= 3) {
                $salary = 100;
            }

            if ($salary === 0) continue;

            // Check if already paid this Saturday
            $alreadyPaid = UserLedger::where('user_id', $user->id)
                ->where('reason', 'weekly_salary')
                ->whereBetween('created_at', [
                    $today->copy()->startOfDay(),
                    $today->copy()->endOfDay()
                ])
                ->exists();

            if ($alreadyPaid) continue;

            // Credit salary
            $user->balance += $salary;
            $user->total_income += $salary;
            $user->team_income += $salary;
            $user->save();

            UserLedger::create([
                'user_id' => $user->id,
                'reason' => 'weekly_salary',
                'amount' => $salary,
                'credit' => $salary,
                'debit' => 0,
                'status' => 'approved',
                'claimed' => true,
                'date' => now()->format('d-m-Y H:i'),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $this->info("User ID {$user->id} received ₦{$salary}");
        }

        $this->info("Weekly salary process completed.");
    }
}
