<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Mail\RegistrationMail;
use App\Models\Admin;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use App\Providers\RouteServiceProvider;
use Carbon\Carbon;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    public function create(Request $request)
    {
        $captcha_code = rand(00000, 99999);
        $ref_by = $request->query('ref');
        return view('app.auth.registration', compact('ref_by', 'captcha_code'));
    }

    public function store(Request $request)
    {
        // Validate input
        $validate = Validator::make($request->all(), [
            'phone' => ['required', 'numeric', 'unique:users,phone'],
            'password' => ['required'],
            //'withdraw_password' => ['required'],
        ]);

        if ($validate->fails()) {
            return back()->withErrors($validate)->withInput();
        }

        $getIp = $request->ip();
        $referrerBonus = '5';  
        $registrationBonus = setting('registration_bonus', 10);  

        // Referral logic
        if ($request->ref_by) {
            $referrer = User::where('ref_id', $request->ref_by)->first();
            if ($referrer) {
                $firstLevelUsers = User::where('ref_by', $referrer->ref_id)->count();
                
                if ($firstLevelUsers <= setting('total_member_register_reword', 10)) {
                    $referrer->balance += $referrerBonus;
                    $referrer->save();

                    DB::table('user_ledgers')->insert([
                        'user_id' => $referrer->id,
                        'reason' => 'referral_bonus',
                        'amount' => $referrerBonus,
                        'credit' => $referrerBonus,
                        'debit' => 0,
                        'status' => 'approved',
                        'date' => Carbon::now(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        }

        // Create the new user
        $user = User::create([
            'name' => 'User' . rand(22, 99),
            'username' => 'uname' . $request->phone,
            'ref_id' => $this->generateUniqueRefCode(),
            'ref_by' => $request->ref_by ?? null, 
            'email' => 'user' . rand(11111, 99999) . time() . '@gmail.com',
            'password' => Hash::make($request->password),
            'type' => 'user',
            'withdraw_password' => $request->withdraw_password,
            'phone' => $request->phone,
            'balance' => $registrationBonus,
            'phone_code' => '+234',
            'ip' => $getIp,
            'remember_token' => Str::random(30),
        ]);

        if ($user) {
            // Add registration bonus to ledger
            DB::table('user_ledgers')->insert([
                'user_id' => $user->id,
                'reason' => 'reg_bonus',
                'amount' => $registrationBonus,
                'credit' => $registrationBonus,
                'debit' => 0,
                'status' => 'approved',
                'date' => Carbon::now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // After user is created, auto-purchase the free plan
            $this->assignFreePackage($user);

            // Log the user in
            Auth::login($user);
            return redirect()->route('dashboard');
        } else {
            return back()->with('message', 'Registration Failed');
        }
    }

    private function generateUniqueRefCode()
    {
        do {
            $refCode = strtoupper(Str::random(6));  
        } while (User::where('ref_id', $refCode)->exists());  
        return $refCode;
    }

    private function assignFreePackage($user)
    {
        // Get the free package (assuming it's defined as price = 0)
        $freePackage = Package::where('price', 0)->first();
        
        if ($freePackage) {
            // Assume the user is assigned the free package after registration
            \DB::transaction(function () use ($user, $freePackage) {
                // Create the purchase record for the free package
                $purchase = new Purchase();
                $purchase->user_id = $user->id;
                $purchase->package_id = $freePackage->id;
                $purchase->amount = 0;
                $purchase->daily_income = $freePackage->commission_with_avg_amount / $freePackage->validity;
                $purchase->date = now()->addHours(24);
                $purchase->validity = now()->addDays($freePackage->validity);
                $purchase->status = 'active';
                $purchase->save();

                // Create ledger entry for the free package purchase
                $ledger = new UserLedger();
                $ledger->user_id = $user->id;
                $ledger->reason = 'product_purchase';
                $ledger->perticulation = 'Free Package Assigned';
                $ledger->amount = 0;
                $ledger->debit = 0;
                $ledger->status = 'approved';
                $ledger->step = 'first';
                $ledger->date = now();
                $ledger->save();
            });
        }
    }

    public function refreshCaptcha()
    {
        return response()->json(['captcha' => captcha_img()]);
    }
}