<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Models\OtpVerification;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log; 

class ForgotPasswordController extends Controller
{
    public function showOtpRequestForm()
    {
        Log::info("Show OTP request form accessed.");
        return view('app.main.forgot-password');
    }

    public function requestOtp(Request $request)
    {
        $request->validate(['phone' => 'required|numeric|exists:users,phone']);

        Log::info("OTP request initiated for phone: " . $request->phone);

        $otp = rand(1000, 9999);

        DB::table('otp_verifications')->updateOrInsert(
            ['phone' => $request->phone],
            ['otp' => $otp, 'created_at' => now(), 'updated_at' => now(), 'verified' => false]
        );

        Log::info("OTP generated and stored for phone: " . $request->phone);

        $message = "Your OTP is $otp";
        $this->sendOtpToPhone($request->phone, $message);

        return response()->json(['message' => 'OTP sent successfully']);
    }

    public function verifyOtp(Request $request)
    {
        $request->validate(['phone' => 'required|numeric', 'otp' => 'required|numeric']);

        Log::info("OTP verification attempt for phone: " . $request->phone);

        $otpRecord = OtpVerification::where('phone', $request->phone)
                                     ->where('otp', $request->otp)
                                     ->first();

        if ($otpRecord && !$otpRecord->verified) {
            $token = Str::random(60); 
            $otpRecord->update(['verified' => true, 'token' => $token]);

            Log::info("OTP verified successfully for phone: " . $request->phone . " with token: " . $token);

            return response()->json(['message' => 'OTP verified successfully', 'token' => $token]);
        }

        Log::warning("Invalid or already verified OTP for phone: " . $request->phone);
        return response()->json(['message' => 'Invalid OTP'], 400);
    }

   public function updatePassword(Request $request)
{
    Log::info('Incoming Password Reset Request:', $request->all());

    $request->validate([
        'phone' => 'required|exists:users,phone',
        'password' => 'required|string|min:4|confirmed',
        'token' => 'required'
    ]);

    $otpRecord = OtpVerification::where('phone', $request->phone)
                                 ->where('verified', true)
                                 ->where('token', $request->token)
                                 ->first();

    if (!$otpRecord) {
        Log::error('Invalid or expired token for phone: ' . $request->phone);
        return response()->json(['message' => 'OTP not verified or expired'], 401);
    }

    $otpExpiryTime = $otpRecord->created_at->addMinutes(10); 
    if (now()->greaterThan($otpExpiryTime)) {
        Log::warning('OTP expired for phone: ' . $request->phone);
        return response()->json(['message' => 'OTP expired'], 401); 
    }

    $user = User::where('phone', $request->phone)->first();
    if (!$user) {
        Log::error('User not found for phone: ' . $request->phone);
        return response()->json(['message' => 'User not found'], 404);
    }

    $user->password = Hash::make($request->password);
    $user->save();

    Log::info('Password updated successfully for phone: ' . $request->phone);

    return response()->json(['message' => 'Password updated successfully']);
}



    private function sendOtpToPhone($phone, $message)
    {
        Log::info("Sending OTP to phone: " . $phone);

        $fields = [
            "sender_id" => "FAST2SMS",
            "message" => $message,
            "language" => "english",
            "route" => "p",
            "numbers" => $phone,
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://www.fast2sms.com/dev/bulkV2",  
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($fields),  
            CURLOPT_HTTPHEADER => [
                "authorization: NvX0ds2LFtrHyYlZ2KtQ2Qr3fq54LxK5yod0JuL7ncXMZatyAnCtX465q0tK", 
                "accept: */*",
                "cache-control: no-cache",
                "content-type: application/json"
            ],
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            Log::error("Error sending OTP to phone " . $phone . ": " . $err);
        } else {
            Log::info("OTP sent successfully to phone: " . $phone);
        }

        return !$err;
    }
}
