<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\UserLedger;
use Illuminate\Support\Facades\Log;

class IncomeController extends Controller
{
    /**
     * Update the total income for users whose current total_income is zero.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateTotalIncome()
{
    try {
        // Define the reasons to filter by
        $incomeReasons = [
            'daily_income',
            'commission',
            'my_commission',
            'check_in_reward',
            'income_commission',
            'referral_bonus'
        ];

        // Process users in chunks
        User::where('total_income', 0) // Only process users with total_income = 0
            ->chunk(500, function ($users) use ($incomeReasons) {
                foreach ($users as $user) {
                    // Calculate total income for the user
                    $totalIncome = UserLedger::where('user_id', $user->id)
                        ->whereIn('reason', $incomeReasons)
                        ->sum('amount');

                    // Update the user's total income if greater than 0
                    if ($totalIncome > 0) {
                        $user->update(['total_income' => $totalIncome]);
                    }
                }
            });

        return response()->json([
            'message' => 'Total income updated successfully for eligible users.',
            'status' => true
        ]);

    } catch (\Exception $e) {
        // Log the error for debugging
        Log::error('Error updating total income: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'message' => 'An error occurred while updating total income. Please check the logs for details.',
            'status' => false
        ], 500);
    }
}
    
public function updateProducts()
{
    try {
        $incomeReason = 'daily_income';

        // Fetch user incomes grouped by user_id
        $incomes = UserLedger::where('reason', $incomeReason)
            ->selectRaw('user_id, SUM(amount) as total_income')
            ->groupBy('user_id')
            ->get();

        // Map user_id to total_income for easy updates
        $userIncomeMap = $incomes->pluck('total_income', 'user_id')->toArray();

        // Update users in chunks for efficiency
        foreach (array_chunk($userIncomeMap, 500, true) as $chunk) {
            foreach ($chunk as $userId => $totalIncome) {
                User::where('id', $userId)->update(['product_income' => $totalIncome]);
            }
        }

        // Return a success response
        return response()->json([
            'message' => 'Product income updated successfully for all users.',
            'status' => true
        ], 200);

    } catch (\Exception $e) {
        // Return an error response in case of an exception
        return response()->json([
            'message' => 'An error occurred while updating product income: ' . $e->getMessage(),
            'status' => false
        ], 500);
    }
}


public function teamIncome()
{
    try {
        // Define the reasons to filter by
        $incomeReasons = ['commission', 'my_commission', 'referral_bonus'];

        // Fetch total income grouped by user_id for the specified reasons
        $incomes = UserLedger::whereIn('reason', $incomeReasons)
            ->selectRaw('user_id, SUM(amount) as total_income')
            ->groupBy('user_id')
            ->get();

        // Create a mapping of user_id => total_income
        $userIncomeMap = $incomes->pluck('total_income', 'user_id')->toArray();

        // Update users in chunks for better performance
        foreach (array_chunk($userIncomeMap, 500, true) as $chunk) {
            foreach ($chunk as $userId => $totalIncome) {
                User::where('id', $userId)->update(['team_income' => $totalIncome]);
            }
        }

        return response()->json([
            'message' => 'Team income updated successfully for all users.',
            'status' => true
        ], 200);

    } catch (\Exception $e) {
        Log::error('Error updating team income: ' . $e->getMessage());
        return response()->json([
            'message' => 'An error occurred while updating team income.',
            'status' => false
        ], 500);
    }
}

public function updateAllUserTeamCounts()
{
    $updatedUsers = 0; // Counter for updated users
    $totalChunks = 0; // Counter for processed chunks

    // Fetch users where any team count field is 0 and process in chunks of 500
    User::where('lv1_team', 0)
        ->orWhere('lv2_team', 0)
        ->orWhere('lv3_team', 0)
        ->chunk(500, function ($users) use (&$updatedUsers, &$totalChunks) {
            $totalChunks++; // Increment chunk counter

            foreach ($users as $user) {
                // Fetch first-level team members (direct referrals)
                $firstLevelUsers = User::where('ref_by', $user->id)->pluck('id'); // Collect IDs for Level 1
                $firstLevelCount = $firstLevelUsers->count();

                // Fetch second-level team members (referred by Level 1)
                $secondLevelUsers = User::whereIn('ref_by', $firstLevelUsers)->pluck('id'); // Collect IDs for Level 2
                $secondLevelCount = $secondLevelUsers->count();

                // Fetch third-level team members (referred by Level 2)
                $thirdLevelUsers = User::whereIn('ref_by', $secondLevelUsers)->pluck('id'); // Collect IDs for Level 3
                $thirdLevelCount = $thirdLevelUsers->count();

                // Calculate total team count
                $totalTeamCount = $firstLevelCount + $secondLevelCount + $thirdLevelCount;

                // Update the user's team stats
                $user->update([
                    'lv1_team' => $firstLevelCount,
                    'lv2_team' => $secondLevelCount,
                    'lv3_team' => $thirdLevelCount,
                    'total_team' => $totalTeamCount,
                ]);

                $updatedUsers++; // Increment the updated user counter
            }
        });

    return response()->json([
        'message' => 'Team counts updated successfully',
        'total_updated_users' => $updatedUsers,
        'total_chunks_processed' => $totalChunks,
    ]);
}


}