<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\FundInvest;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\Rebate;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class AdminController extends Controller
{
    public function login()
    {
        if (Auth::guard('admin')->check()) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.auth.login');
    }


public function commission()
{
    $admin = Admin::first();

    // Check if salary has already been served today
    if ($admin->salary_date == date('Y-m-d')) {
        return back()->with('error', 'Today Salary Served');
    }

    // Process purchases in batches
    Purchase::where('status', 'active')
        ->chunk(1000, function ($purchases) {
            foreach ($purchases as $purchase) {
                // Check if the associated user exists
                $user = User::find($purchase->user_id);
                if (!$user) {
                    Log::warning('Purchase skipped: User not found', ['purchase_id' => $purchase->id, 'user_id' => $purchase->user_id]);
                    continue;
                }

                // Check if the associated package exists
                $package = Package::find($purchase->package_id);
                if (!$package) {
                    Log::warning('Purchase skipped: Package not found', ['purchase_id' => $purchase->id, 'package_id' => $purchase->package_id]);
                    continue;
                }

                // Validate date condition
                if (now()->greaterThanOrEqualTo($purchase->date)) {
                    $dailyIncome = $purchase->daily_income;

                    // Distribute commission
                    $this->distributeCommission($user, $dailyIncome);

                    // Update user's receivable amount
                    $user->receive_able_amount += $dailyIncome;
                    $user->save();

                    // Update purchase date for the next cycle
                    $purchase->date = now()->addHours(24);
                    $purchase->save();

                    // Check if the purchase has expired
                    $checkExpire = new Carbon($purchase->validity);
                    if ($checkExpire->isPast()) {
                        $purchase->status = 'inactive';
                        $purchase->save();
                        Log::info('Purchase marked as expired', ['purchase_id' => $purchase->id]);
                    }

                    Log::info('Commission distributed', [
                        'user_id' => $user->id,
                        'purchase_id' => $purchase->id,
                        'daily_income' => $dailyIncome,
                    ]);
                } else {
                    Log::info('Purchase skipped due to date condition', [
                        'purchase_id' => $purchase->id,
                        'current_date' => now(),
                        'purchase_date' => $purchase->date,
                    ]);
                }
            }
        });

    Log::info('Commission process completed');
}


public function distributeCommission($user, $dailyIncome)
{
    $rebates = Rebate::first();
    if (!$rebates) {
        return; 
    }

    $commissionRates = [
        'first' => $rebates->income_commission1 / 100,  
        'second' => $rebates->income_commission2 / 100, 
        'third' => $rebates->income_commission3 / 100   
    ];

    $levels = [
        'first' => $user->referrer, 
        'second' => $user->referrer->referrer ?? null, 
        'third' => $user->referrer->referrer->referrer ?? null 
    ];

    foreach ($levels as $level => $referrer) {
        if ($referrer && isset($commissionRates[$level])) {

            $commissionAmount = $dailyIncome * $commissionRates[$level];

            // Increment balance and other fields
            $referrer->increment('balance', $commissionAmount);
            $referrer->increment('total_income', $commissionAmount);
            $referrer->increment('today_income', $commissionAmount);
            $referrer->increment('team_income', $commissionAmount);
            $referrer->save();
            
            // Save the referrer model (this is optional, as increment already updates the database)


            // Create ledger for the commission
            $ledger = new UserLedger();
            $ledger->user_id = $referrer->id;
            $ledger->get_balance_from_user_id = $user->id;
            $ledger->reason = 'income_commission';
            $ledger->perticulation = "{$level} level commission from user {$user->id}";
            $ledger->amount = $commissionAmount;
            $ledger->credit = $commissionAmount;
            $ledger->status = 'approved';
            $ledger->date = date('d-m-Y H:i');
            $ledger->step = $level;
            $ledger->save();
        }
    }


    $userFunds = FundInvest::where('status', 'active')->get();
    foreach ($userFunds as $fund) {
        $xTime = Carbon::parse($fund->validity_expired);
        if ($xTime->isPast()) {
            if ($fund->status == 'active') {
                // Update FundInvest Record
                $fIn = FundInvest::where('id', $fund->id)->first();
                $fIn->status = 'inactive';

                // Insert Balance into user balance
                if ($fIn->save()) {
                    $user = User::where('id', $fund->user_id)->first();
                    $user->balance = $user->balance + $fund->return_amount;
                    $user->save();

                    // Ledger creation for FundInvest remains intact
                    $ledger = new UserLedger();
                    $ledger->user_id = $user->id;
                    $ledger->reason = 'fund';
                    $ledger->perticulation = 'Congratulations You have received your fund with commission.';
                    $ledger->amount = $fund->return_amount;
                    $ledger->debit = $fund->return_amount;
                    $ledger->status = 'approved';
                    $ledger->date = date('d-m-Y H:i');
                    $ledger->save();
                }
            }
        }
    }
}


    public function login_submit(Request $request)
    {
        $credentials = $request->only('email', 'password');
        if (Auth::guard('admin')->attempt($credentials)) {
            $admin = Auth::guard('admin')->user();
            if ($admin->type == 'admin') {
                return redirect()->route('admin.dashboard')->with('success', 'Logged In Successful.');
            } else {
                return error_redirect('admin.login', 'error', 'Admin Credentials Very Secured Please Don"t try Again.');
            }
        } else {
            return error_redirect('admin.login', 'error', 'Admin Credentials Does Not Match.');
        }
    }

    public function logout()
    {
        if (Auth::guard('admin')->check()) {
            Auth::guard('admin')->logout();
            return redirect()->route('admin.login')->with('success', 'Logged out successful.');
        } else {
            return error_redirect('admin.login', 'error', 'You are already logged out.');
        }
    }

    public function dashboard()
    {
        return view('admin.dashboard');
    }

    public function developer()
    {
        return view('admin.developer');
    }

    public function profile()
    {
        return view('admin.profile.index');
    }

    public function profile_update()
    {
        $admin = Admin::first();
        return view('admin.profile.update-details', compact('admin'));
    }

    public function profile_update_submit(Request $request)
    {
        $admin = Admin::find(1);
        $path = uploadImage(false, $request, 'photo', 'admin/assets/images/profile/', $admin->photo);
        $admin->photo = $path ?? $admin->photo;
        $admin->name = $request->name;
        $admin->email = $request->email;
        $admin->phone = $request->phone;
        $admin->address = $request->address;
        $admin->update();
        return redirect()->route('admin.profile.update')->with('success', 'Admin profile updated.');
    }

    public function change_password()
    {
        $admin = admin()->user();
        return view('admin.profile.change-password', compact('admin'));
    }

    public function check_password(Request $request)
    {
        $admin = admin()->user();
        $password = $request->password;
        if (Hash::check($password, $admin->password)) {
            return response()->json(['message' => 'Password matched.', 'status' => true]);
        } else {
            return response()->json(['message' => 'Password dose not match.', 'status' => false]);
        }
    }

    public function change_password_submit(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'old_password' => 'required',
            'new_password' => 'required',
            'confirm_password' => 'required'
        ]);
        if ($validate->fails()) {
            session()->put('errors', true);
            return redirect()->route('admin.changepassword')->withErrors($validate->errors());
        }

        $admin = admin()->user();
        $password = $request->old_password;
        if (Hash::check($password, $admin->password)) {
            if (strlen($request->new_password) > 5 && strlen($request->confirm_password) > 5) {
                if ($request->new_password === $request->confirm_password) {
                    $admin->password = Hash::make($request->new_password);
                    $admin->update();
                    return redirect()->route('admin.changepassword')->with('success', 'Password changed successfully');
                } else {
                    return error_redirect('admin.changepassword', 'error', 'New password and confirm password dose not match');
                }
            } else {
                return error_redirect('admin.changepassword', 'error', 'Password must be greater then 6 or equal.');
            }
        } else {
            return error_redirect('admin.changepassword', 'error', 'Password dose not match');
        }
    }
}
