<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Withdrawal;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class IpnQePayController extends Controller
{
    public function ipnDeposit(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'mchOrderNo' => 'required|string',
                'orderNo'    => 'required|string',
                'amount'     => 'required',
                'tradeResult'=> 'required|in:SUCCESS,1'
            ]);

            if ($validator->fails()) throw new \Exception(implode('...', $validator->errors()->all()));

            $recharge = Deposit::where('order_id', $request->mchOrderNo)->first();

            if (!$recharge) {
                return response(['success' => true, 'status' => 'ok', 'message' => ['success' => 'Transaction reference not found']], 200);
            }

            $updateData = self::updateDeposit($request->mchOrderNo, $request->amount, $request->all());

            if ($updateData instanceof \Exception) throw new \Exception($updateData->getMessage());

            return 'success';
        } catch (\Exception $th) {
            return response(['success' => true, 'status' => 'error', 'message' => ['error' => $th->getMessage()]], 400);
        }
    }

    public static function updateDeposit(string $reference, string $amount, array $data)
    {
        try {
            $transaction = Deposit::where('order_id', $reference)->first();
            if (!$transaction) throw new \Exception('Transaction not found');

            if (in_array($transaction->status, ['approved', 'rejected'])) {
                throw new \Exception('Transaction already complete');
            }

            $transaction->detail = $data;
            $transaction->status = 'approved';
            $transaction->transaction_id = $data['orderNo'] ?? null;
            $transaction->save();

            $user = User::find($transaction->user_id);
            $user->balance += $transaction->amount;
            $user->save();

            return $transaction;
        } catch (\Exception $th) {
            return $th;
        }
    }

    public function ipnTransfer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'merTransferId'   => 'required|string',
                'tradeNo'         => 'required|string',
                'transferAmount'  => 'required',
                'tradeResult'     => 'required|in:SUCCESS,FAIL,1,2'
            ]);

            if ($validator->fails()) throw new \Exception(implode('...', $validator->errors()->all()));

            $payout = Withdrawal::where('trx', $request->merTransferId)->first();

            if (!$payout) {
                return response(['success' => true, 'status' => 'ok', 'message' => ['success' => 'Transaction reference not found']], 200);
            }

            $updateData = self::updatePayout($request->tradeResult, $request->merTransferId, $request->transferAmount, $request->all());

            if ($updateData instanceof \Exception) throw new \Exception($updateData->getMessage());

            return 'success';
        } catch (\Exception $th) {
            return response(['success' => true, 'status' => 'error', 'message' => ['error' => $th->getMessage()]], 400);
        }
    }

    public static function updatePayout(string $status, string $reference, string $amount, array $data)
    {
        try {
            $payout = Withdrawal::where('trx', $reference)->first();
            if (!$payout) throw new \Exception('Transaction not found');

            $user = User::find($payout->user_id);
            if (!$user) throw new \Exception('User not found');

            // Prevent double update if already failed or approved
            if (in_array($payout->status, ['rejected', 'approved'])) {
                throw new \Exception('Transaction already complete');
            }

            // Save transaction ID (from gateway)
            $payout->transaction_id = $data['tradeNo'] ?? null;

            // Convert status
            $normalized = strtolower($status);
            if (in_array($normalized, ['success', '1'])) {
                $payout->status = 'approved';
                $payout->save();
            } elseif (in_array($normalized, ['fail', '2'])) {
                $payout->status = 'rejected';
                $payout->save();

                // Refund balance
                $user->balance += $payout->amount;
                $user->save();
            } else {
                // If neither success nor fail, mark pending
                $payout->status = 'pending';
                $payout->save();
            }

            return $payout;
        } catch (\Exception $th) {
            return $th;
        }
    }
}
