<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\Rebate;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    public function purchaseConfirmation($id, $packageType)
    {
        // Placeholder for confirmation logic
    }

    public function buyPackage($id)
    {
        $package = Package::find($id);
        $user = Auth::user();
        $rebate = Rebate::first();

        if (!$package) {
            session()->flash('error', 'Package not found.');
            return redirect()->back();
        }

        if ($package->status !== 'active') {
            session()->flash('error', 'This package is inactive.');
            return redirect()->back();
        }

        if ($package->is_sold_out) {
            session()->flash('error', 'This package is currently sold out.');
            return redirect()->back();
        }

        // Handle purchase limit
        $existingPurchaseCount = Purchase::where('user_id', $user->id)
            ->where('package_id', $package->id)
            ->where('status', 'active')
            ->count();

        if ($package->buy_limit > 0 && $existingPurchaseCount >= $package->buy_limit) {
            session()->flash('error', 'You have reached the purchase limit for this package.');
            return redirect()->back();
        }

        // Check recharge history if required
        if ($package->price > 0) {
            $totalDeposit = Deposit::where('user_id', $user->id)
                ->where('status', 'approved')
                ->sum('amount');

            if ($totalDeposit < 300) {
                session()->flash('error', 'Minimum Recharge Required ₱300.');
                return redirect()->back();
            }
        }

        // Determine actual price (consider discount)
        $finalPrice = $package->discounted_price ?? $package->price;

        if ($finalPrice <= 0) {
            return $this->purchaseFreePackage($user, $package, $rebate);
        }

        if ($finalPrice > $user->deposit_wallet) {
            session()->flash('error', 'Insufficient balance to purchase this package.');
            return redirect()->back();
        }

        return $this->purchasePaidPackage($user, $package, $rebate, $finalPrice);
    }

    protected function purchaseFreePackage($user, $package, $rebate)
    {
        DB::transaction(function () use ($user, $package, $rebate) {
            $purchase = new Purchase();
            $purchase->user_id = $user->id;
            $purchase->package_id = $package->id;
            $purchase->amount = 0;
            $purchase->daily_income = $package->commission_with_avg_amount / $package->validity;
            $purchase->date = now()->addHours(24);
            $purchase->validity = now()->addDays($package->validity);
            $purchase->status = 'active';
            $purchase->save();

            UserLedger::create([
                'user_id' => $user->id,
                'reason' => 'product_purchase',
                'perticulation' => 'Free Package Added.',
                'amount' => 0,
                'debit' => 0,
                'status' => 'approved',
                'step' => 'first',
                'date' => now(),
            ]);

            $this->handleReferrals($user, $package, $rebate);
        });

        session()->flash('success', 'Free package purchased successfully!');
        return redirect()->back();
    }

    protected function purchasePaidPackage($user, $package, $rebate, $price)
    {
        DB::transaction(function () use ($user, $package, $rebate, $price) {
            // Deduct wallet
            $user->decrement('deposit_wallet', $price);
            $user->update(['investor' => 1]);

            $purchase = new Purchase();
            $purchase->user_id = $user->id;
            $purchase->package_id = $package->id;
            $purchase->amount = $price;
            $purchase->daily_income = $package->commission_with_avg_amount / $package->validity;
            $purchase->date = now()->addHours(24);
            $purchase->validity = now()->addDays($package->validity);
            $purchase->status = 'active';
            $purchase->save();

            UserLedger::create([
                'user_id' => $user->id,
                'reason' => 'product_purchase',
                'perticulation' => 'Package Purchased',
                'amount' => $price,
                'debit' => $price,
                'status' => 'approved',
                'step' => 'first',
                'date' => now(),
            ]);

            $this->handleReferrals($user, $package, $rebate);
        });

        session()->flash('success', 'Package purchased successfully!');
        return redirect()->back();
    }

    protected function handleReferrals($user, $package, $rebate)
    {
        // Check if the user has any previous purchases
        $previousPurchaseCount = Purchase::where('user_id', $user->id)->count();

        if ($previousPurchaseCount > 0) {
            return; // Skip referral handling if it's not the first purchase
        }

        $referralLevels = [
            ['level' => 'first', 'rate' => $rebate->interest_commission1],
            ['level' => 'second', 'rate' => $rebate->interest_commission2],
            ['level' => 'third', 'rate' => $rebate->interest_commission3],
        ];

        $currentRef = User::where('ref_id', $user->ref_by)->first();

        foreach ($referralLevels as $ref) {
            if (!$currentRef) break;

            $amount = $package->price * $ref['rate'] / 100;

            $currentRef->increment('balance', $amount);
            $currentRef->increment('total_income', $amount);
            $currentRef->increment('today_income', $amount);
            $currentRef->increment('team_income', $amount);

            UserLedger::create([
                'user_id' => $currentRef->id,
                'get_balance_from_user_id' => $user->id,
                'reason' => 'commission',
                'perticulation' => ucfirst($ref['level']) . ' Level Commission Received',
                'amount' => $amount,
                'credit' => $amount,
                'status' => 'approved',
                'step' => $ref['level'],
                'date' => now()->format('d-m-Y H:i'),
            ]);

            $currentRef = User::where('ref_id', $currentRef->ref_by)->first();
        }
    }

    public function vip_confirm($vip_id)
    {
        $vip = Package::find($vip_id);
        return view('app.main.vip_confirm', compact('vip'));
    }

    protected function ref_user($ref_by)
    {
        return User::where('ref_id', $ref_by)->first();
    }
}
